<?php
/**
 * Dashboard Widget: بروزرسانی از مخزن وردپرس VPS (بدون کش؛ IP-first access; همه‌چیز با فونت وزیر)
 * File: repository-admin.php
 */
if (!defined('ABSPATH')) exit;

/* ---------- Register Widget ---------- */
add_action('wp_dashboard_setup', function () {
    wp_add_dashboard_widget(
        'vps7_repo_updates_widget',
        'بروزرسانی از مخزن وردپرس VPS 7',
        'vps7_repo_render_widget'
    );
});

/* ---------- Admin Ajax ---------- */
add_action('wp_ajax_vps7_repo_widget_data', 'vps7_repo_ajax_widget_data');

/* ---------- Helpers ---------- */
function vps7_repo_api_root() {
    if (class_exists('VPS7_Repository_Client')) {
        return rtrim(VPS7_Repository_Client::SERVER_BASE, '/') . '/' . ltrim(VPS7_Repository_Client::API_BASE, '/');
    }
    return 'https://vps7.net/wp-json/vps7/v1';
}
function vps7_repo_license_opt_key() {
    return class_exists('VPS7_Repository_Client') ? VPS7_Repository_Client::OPT_LICENSE : 'vps7_client_license';
}
function vps7_repo_remote_get($url, $args = array()) {
    // هر بار تازه از سرور بگیر
    $args['nocache'] = uniqid('', true);
    $url = add_query_arg($args, $url);
    $r = wp_remote_get($url, array(
        'timeout' => 20,
        'headers' => array('Cache-Control'=>'no-cache, no-store, max-age=0')
    ));
    if (is_wp_error($r)) return array();
    $b = wp_remote_retrieve_body($r);
    $d = json_decode($b, true);
    return is_array($d) ? $d : array();
}

/** ✅ دسترسی «IP-اول» و سخت‌گیرانه (بدون کش) — دقیقاً مثل vps7-client-updates.php */
function vps7_repo_check_access_strict(){
    $api     = vps7_repo_api_root();
    $opt_key = vps7_repo_license_opt_key();
    $license = trim(get_option($opt_key, ''));
    $domain  = parse_url(site_url(), PHP_URL_HOST);

    // 1) پروب IP (هر بار)
    $probe = vps7_repo_remote_get($api.'/access', array(
        'intent'     => 'connect',
        'consume'    => 0,
        'require_ip' => 1,
        'domain'     => $domain,
    ));
    if (is_array($probe) && !empty($probe['allowed']) && strtolower((string)($probe['via'] ?? 'ip')) === 'ip') {
        return array('allowed'=>true, 'via'=>'ip');
    }

    // 2) اگر IP مجاز نبود، لایسنس را *در لحظه* چک کن
    if ($license !== '') {
        $acc = vps7_repo_remote_get($api.'/access', array(
            'license' => $license,
            'domain'  => $domain,
            'intent'  => 'connect',
            'consume' => 0,
        ));
        if (is_array($acc) && (!empty($acc['fatal']) || !empty($acc['domain_unbound']) || !empty($acc['invalidate']) || !empty($acc['should_unset']))) {
            delete_option($opt_key);
            return array('allowed'=>false, 'via'=>'license');
        }
        if (is_array($acc) && !empty($acc['allowed'])) {
            return array('allowed'=>true, 'via'=>'license');
        }
        return array('allowed'=>false, 'via'=>'license');
    }

    // هیچ‌کدام
    return array('allowed'=>false, 'via'=>'ip');
}

/** جمع‌آوری نسخه‌های لوکال برای مچ دقیق (بدون کش) */
function vps7_repo_collect_local_versions_rich(){
    require_once ABSPATH.'wp-admin/includes/plugin.php';
    $norm = function($s){ return preg_replace('/[\-_]+/','', sanitize_title((string)$s)); };

    // plugins
    $p_map = array(); // key => version
    foreach (get_plugins() as $file=>$data){
        $folder = explode('/', $file)[0];
        $ver    = isset($data['Version']) ? (string)$data['Version'] : '';
        $name   = isset($data['Name']) ? (string)$data['Name'] : '';
        $td     = isset($data['TextDomain']) ? (string)$data['TextDomain'] : '';

        $keys = array(
            sanitize_title($folder),
            sanitize_title($name),
            sanitize_title($td),
        );
        foreach ($keys as $k){
            if (!$k) continue;
            $p_map[$k] = $ver;
            $p_map[$norm($k)] = $ver;
        }
    }

    // themes
    $t_map = array();
    foreach (wp_get_themes() as $ss=>$th){
        $ver  = (string)$th->get('Version');
        $name = (string)$th->get('Name');
        $keys = array(
            sanitize_title($ss),
            sanitize_title($name),
        );
        foreach ($keys as $k){
            if (!$k) continue;
            $t_map[$k] = $ver;
            $t_map[$norm($k)] = $ver;
        }
    }
    return array($p_map, $t_map);
}

/** دریافت مپ سرور (بدون هیچ ترنزینت/کش) */
function vps7_repo_fetch_server_maps_widget_live(){
    $api = vps7_repo_api_root();

    // یکباره زیاد می‌گیریم؛ اما هر بار تازه
    $pl = vps7_repo_remote_get($api . '/plugins', array('per_page'=>300));
    $th = vps7_repo_remote_get($api . '/themes',  array('per_page'=>300));

    $p_items = isset($pl['items']) && is_array($pl['items']) ? $pl['items'] : array();
    $t_items = isset($th['items']) && is_array($th['items']) ? $th['items'] : array();

    $norm = function($s){ return preg_replace('/[\-_]+/','', sanitize_title((string)$s)); };

    $p_map = array();
    foreach ($p_items as $it){
        $title = (string)($it['title'] ?? '');
        $slug0 = (string)($it['slug']  ?? $title);
        $slug  = sanitize_title($slug0);
        if (!$slug) continue;
        $data = array(
            'version' => (string)($it['version'] ?? ''),
            'title'   => $title ?: $slug,
            'id'      => intval($it['id'] ?? 0),
            'slug'    => $slug,
            'type'    => 'plugin',
        );
        $p_map[$slug] = $data;
        $p_map[$norm($slug)] = $data;
    }

    $t_map = array();
    foreach ($t_items as $it){
        $title = (string)($it['title'] ?? '');
        $slug0 = (string)($it['slug']  ?? $title);
        $slug  = sanitize_title($slug0);
        if (!$slug) continue;
        $data = array(
            'version' => (string)($it['version'] ?? ''),
            'title'   => $title ?: $slug,
            'id'      => intval($it['id'] ?? 0),
            'slug'    => $slug,
            'type'    => 'theme',
        );
        $t_map[$slug] = $data;
        $t_map[$norm($slug)] = $data;
    }

    return array('plugins'=>$p_map, 'themes'=>$t_map);
}

/** ---------- AJAX payload (بدون کش) ---------- */
function vps7_repo_ajax_widget_data(){
    if (!current_user_can('read')) wp_send_json_error('forbidden');

    // فقط برای نصب در کلیک دکمه به درد می‌خورد
    $nonce = isset($_POST['nonce']) ? sanitize_text_field($_POST['nonce']) : '';
    if (!wp_verify_nonce($nonce, 'vps7_nonce')) {
        // نمایش صرف؛ نصب جداگانه nonce چک می‌شود
    }

    $acc     = vps7_repo_check_access_strict();
    $allowed = !empty($acc['allowed']);

    list($local_plugins, $local_themes) = vps7_repo_collect_local_versions_rich();
    $server = vps7_repo_fetch_server_maps_widget_live(); // بدون کش

    $cmp = function($a,$b){
        $pa = array_map('intval', explode('.', (string)$a));
        $pb = array_map('intval', explode('.', (string)$b));
        $n = max(count($pa), count($pb));
        for ($i=0; $i<$n; $i++){
            $x = $pa[$i] ?? 0; $y = $pb[$i] ?? 0;
            if ($x > $y) return 1;
            if ($x < $y) return -1;
        }
        return 0;
    };

    $updates_plugins = array(); $seen = array();
    foreach ($local_plugins as $key=>$verLocal){
        if (!isset($server['plugins'][$key])) continue;
        $remote = $server['plugins'][$key];
        $verRemote = $remote['version'];
        $slug      = $remote['slug'];
        if (isset($seen[$slug])) continue; $seen[$slug]=1;
        if ($verRemote && $cmp($verRemote, $verLocal) > 0){
            $updates_plugins[] = array(
                'slug'   => $slug,
                'local'  => $verLocal,
                'remote' => $verRemote,
                'title'  => $remote['title'],
                'id'     => $remote['id'],
            );
        }
    }

    $updates_themes = array(); $seen = array();
    foreach ($local_themes as $key=>$verLocal){
        if (!isset($server['themes'][$key])) continue;
        $remote = $server['themes'][$key];
        $verRemote = $remote['version'];
        $slug      = $remote['slug'];
        if (isset($seen[$slug])) continue; $seen[$slug]=1;
        if ($verRemote && $cmp($verRemote, $verLocal) > 0){
            $updates_themes[] = array(
                'slug'   => $slug,
                'local'  => $verLocal,
                'remote' => $verRemote,
                'title'  => $remote['title'],
                'id'     => $remote['id'],
            );
        }
    }

    $resp = array(
        'allowed'       => $allowed ? 1 : 0,
        'via'           => (string)($acc['via'] ?? ''),
        'plugins'       => array_slice($updates_plugins, 0, 5),
        'themes'        => array_slice($updates_themes,  0, 5),
        'more_plugins'  => max(0, count($updates_plugins) - 5),
        'more_themes'   => max(0, count($updates_themes)  - 5),
        'updates_url'   => admin_url('admin.php?page=vps7_updates'),
        'nonce'         => wp_create_nonce('vps7_nonce'),
    );

    wp_send_json_success($resp);
}

/* ---------- Render (اسکلت + Lazy بدون هیچ کش سمت مرورگر) ---------- */
function vps7_repo_render_widget() {
    // فونت وزیر: غیرمسدودکننده + preconnect
    echo '<link rel="dns-prefetch" href="//vps7.net" />';
    echo '<link rel="preconnect" href="https://vps7.net" crossorigin />';
    echo '<link rel="preload" href="https://cdn.jsdelivr.net/npm/vazirmatn@33.003/Farsi-Digits/font-face.css" as="style" onload="this.onload=null;this.rel=\'stylesheet\'">';
    echo '<noscript><link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/vazirmatn@33.003/Farsi-Digits/font-face.css"></noscript>';

    $ajax_url = admin_url('admin-ajax.php');
    $nonce    = wp_create_nonce('vps7_nonce');

    ?>
    <style>
        /* همه‌جا وزیر */
        #vps7_repo_updates_widget .inside,
        #vps7_repo_updates_widget .handlediv,
        #vps7_repo_updates_widget .hndle,
        .vps7-wdg-wrap,
        .vps7-wdg-wrap * {
            font-family: "Vazirmatn","Vazir",system-ui,-apple-system,BlinkMacSystemFont,"Segoe UI",sans-serif !important;
            direction: rtl;
        }
        .vps7-wdg-table{width:100%;border-collapse:collapse}
        .vps7-wdg-table th,.vps7-wdg-table td{padding:8px 10px;border-bottom:1px solid #e5e7eb;vertical-align:middle;text-align:right}
        .vps7-wdg-table th{background:#f8fafc;font-weight:700}
        .vps7-wdg-actions{text-align:left;white-space:nowrap}

        .vps7-wdg-section-title{margin:12px 0 8px;font-size:14px;font-weight:800}

        .vps7-wdg-footer{margin-top:10px}
        .vps7-wdg-footer .button{margin-left:6px}

        .vps7-skel{position:relative;min-height:54px;overflow:hidden;background:linear-gradient(90deg,#f3f4f6 0,#f9fafb 20%,#f3f4f6 40%);background-size:200% 100%;animation:vps7-shimmer 1.2s infinite}
        @keyframes vps7-shimmer{0%{background-position:200% 0}100%{background-position:-200% 0}}

        /* استایل بنرها هماهنگ با بقیه قسمت‌ها */
        .vps7-infobox{display:flex;gap:12px;align-items:flex-start;border:1px solid #e5e7eb;background:#fff;border-radius:10px;padding:12px 14px;margin:10px 0 14px 0}
        .vps7-ok{border-color:#22c55e;background:#ecfdf5}
        .vps7-err{border-color:#ef4444;background:#fef2f2}
        .vps7-ico{flex:0 0 28px;width:28px;height:28px;border-radius:50%;display:flex;align-items:center;justify-content:center;margin-top:2px}
        .vps7-ok .vps7-ico{background:#22c55e;color:#fff}
        .vps7-err .vps7-ico{background:#ef4444;color:#fff}
        .vps7-infobox h3{margin:0 0 6px 0;font-size:15px}
        .vps7-infobox p{margin:0}
        .vps7-wdg-wrap .button[disabled], .vps7-wdg-wrap .button:disabled{
            opacity:.45!important;filter:grayscale(100%);cursor:not-allowed!important;pointer-events:none;
        }
    </style>

    <div class="vps7-wdg-wrap" id="vps7_repo_widget"
         data-ajax="<?php echo esc_url($ajax_url); ?>"
         data-nonce="<?php echo esc_attr($nonce); ?>">

        <div id="vps7_repo_banner" style="margin:8px 0;"></div>

        <h3 class="vps7-wdg-section-title">پلاگین‌ها</h3>
        <div id="vps7_plugins_box" class="vps7-skel"></div>

        <br><h3 class="vps7-wdg-section-title" style="margin-top:14px">قالب‌ها</h3>
        <div id="vps7_themes_box" class="vps7-skel"></div>

        <div class="vps7-wdg-footer" id="vps7_repo_footer" style="display:none"></div>
    </div>

    <script>
    (function(){
        var cont  = document.getElementById('vps7_repo_widget');
        var ajax  = cont ? cont.getAttribute('data-ajax') : '';
        var nonce = cont ? cont.getAttribute('data-nonce') : '';

        function el(tag, attrs, html){
            var e = document.createElement(tag);
            if (attrs) Object.keys(attrs).forEach(function(k){ e.setAttribute(k, attrs[k]); });
            if (html !== undefined) e.innerHTML = html;
            return e;
        }
        function buildTable(rows, type, allowed){
            var tbl = el('table', {'class':'vps7-wdg-table'});
            var thead = '<thead><tr><th>نام</th><th>ورژن فعلی</th><th>ورژن بروزرسانی</th><th class="vps7-wdg-actions">عملیات</th></tr></thead>';
            var body = '<tbody>';
            rows.forEach(function(r){
                var btn = '<button class="button button-primary vps7-update"'
                        + (allowed? '' : ' disabled aria-disabled="true" title="دسترسی شما مجاز نیست"')
                        + ' data-type="'+type+'" data-id="'+(r.id||0)+'" data-title="'+(r.title||'')+'" data-slug="'+(r.slug||'')+'" data-version="'+(r.remote||'')+'">بروزرسانی</button>';
                body += '<tr>'
                     +  '<td>'+ (r.title||r.slug||'') +'</td>'
                     +  '<td>'+ (r.local||'') +'</td>'
                     +  '<td>'+ (r.remote||'') +'</td>'
                     +  '<td class="vps7-wdg-actions">'+ btn +'</td>'
                     +  '</tr>';
            });
            body += '</tbody>';
            tbl.innerHTML = thead + body;
            return tbl;
        }

        function fetchData(){
            var fd = new FormData();
            fd.append('action','vps7_repo_widget_data');
            fd.append('nonce', nonce);
            // بدون کش سمت مرورگر
            return fetch(ajax, {
                method:'POST',
                credentials:'same-origin',
                cache: 'no-store',
                headers: {'Cache-Control':'no-cache, no-store, max-age=0'},
                body: fd
            })
            .then(function(r){ return r.json(); })
            .then(function(j){
                if (!j || !j.success) throw new Error('bad response');
                return j.data || {};
            });
        }

        function bannerHTML(allowed, via){
            var svg_ok  = '<svg viewBox="0 0 24 24" width="18" height="18" fill="currentColor"><path d="M9 16.2l-3.5-3.5L4 14.2l5 5 11-11-1.5-1.5z"/></svg>';
            var svg_err = '<svg viewBox="0 0 24 24" width="18" height="18" fill="currentColor"><path d="M1 21h22L12 2 1 21zm11-9h2v5h-2zm0 6h2v2h-2z"/></svg>';
            if (allowed && (via === 'ip' || via === 'license')) {
                var title = (via === 'ip') ? 'دسترسی IP شما مجاز است' : 'دسترسی با لایسنس معتبر';
                var desc  = (via === 'ip') ? 'نیازی به ثبت لایسنس نیست.' : 'این دسترسی به‌واسطهٔ لایسنس فعال شده است.';
                return '<div class="vps7-infobox vps7-ok"><div class="vps7-ico">'+svg_ok+'</div><div><h3>'+title+'</h3><p>'+desc+'</p></div></div>';
            }
            var shop_link     = '<?php echo esc_url('https://vps7.net/shop'); ?>';
            var settings_link = '<?php echo esc_url(admin_url('admin.php?page=vps7_settings')); ?>';
            return '<div class="vps7-infobox vps7-err"><div class="vps7-ico">'+svg_err+'</div><div><h3>دسترسی شما مجاز نیست</h3><p>برای فعال‌سازی دسترسی از طریق لایسنس به <a href="'+settings_link+'">بخش لایسنس</a> مراجعه کنید، و برای خرید لایسنس به <a href="'+shop_link+'" target="_blank" rel="noopener">VPS 7</a> مراجعه کنید.</p></div></div>';
        }

        function render(resp){
            var banner = document.getElementById('vps7_repo_banner');
            var pBox   = document.getElementById('vps7_plugins_box');
            var tBox   = document.getElementById('vps7_themes_box');
            var foot   = document.getElementById('vps7_repo_footer');

            var allowed = !!resp.allowed;
            var via     = (resp.via || '');

            banner.innerHTML = bannerHTML(allowed, via);

            pBox.classList.remove('vps7-skel');
            if (resp.plugins && resp.plugins.length){
                pBox.innerHTML = '';
                pBox.appendChild( buildTable(resp.plugins, 'plugin', allowed) );
                if (resp.more_plugins && resp.more_plugins > 0){
                    var a = el('a', {'class':'button', 'href': resp.updates_url}, 'مشاهده همه بروزرسانی‌ها');
                    pBox.appendChild( el('p', null, a.outerHTML) );
                }
            } else {
                pBox.innerHTML = '<p>هیچ بروزرسانی فعالی برای پلاگین‌ها یافت نشد.</p>';
            }

            tBox.classList.remove('vps7-skel');
            if (resp.themes && resp.themes.length){
                tBox.innerHTML = '';
                tBox.appendChild( buildTable(resp.themes, 'theme', allowed) );
                if (resp.more_themes && resp.more_themes > 0){
                    var a2 = el('a', {'class':'button', 'href': resp.updates_url}, 'مشاهده همه بروزرسانی‌ها');
                    tBox.appendChild( el('p', null, a2.outerHTML) );
                }
            } else {
                tBox.innerHTML = '<p>هیچ بروزرسانی فعالی برای قالب‌ها یافت نشد.</p>';
            }

            foot.style.display = 'block';
            foot.innerHTML = '<a class="button button-primary" href="'+resp.updates_url+'">رفتن به صفحه بروزرسانی‌ها</a>';

            // بایند کلیک بروزرسانی (بدون کش)
            if (!cont.__vps7Bound) {
                cont.__vps7Bound = true;
                cont.addEventListener('click', function(ev){
                    var btn = ev.target.closest('.vps7-update');
                    if (!btn) return;
                    ev.preventDefault();
                    if (btn.disabled) return;

                    var id   = btn.getAttribute('data-id');
                    var type = btn.getAttribute('data-type');

                    btn.textContent = 'در حال نصب...';
                    btn.disabled = true;

                    var fd = new FormData();
                    fd.append('action','vps7_install_item');
                    fd.append('nonce', resp.nonce || nonce);
                    fd.append('id', id);
                    fd.append('type', type);

                    fetch('<?php echo esc_url(admin_url('admin-ajax.php')); ?>?t='+(Date.now()), {
                        method:'POST',
                        credentials:'same-origin',
                        cache: 'no-store',
                        headers: {'Cache-Control':'no-cache, no-store, max-age=0'},
                        body: fd
                    })
                    .then(function(r){ return r.json(); })
                    .then(function(j){
                        if (j && j.success){
                            btn.textContent = 'بروزرسانی شد';
                            btn.title = 'نصب/آپدیت انجام شد';

                            // فوراً دادهٔ تازه بگیر و باز رندر کن (بدون کش)
                            fetchData().then(render).catch(function(){ /* سکوت */ });
                        } else {
                            btn.disabled = false;
                            btn.textContent = 'بروزرسانی';
                            alert('خطا: ' + (j && j.data ? j.data : 'نامشخص'));
                        }
                    })
                    .catch(function(){
                        btn.disabled = false;
                        btn.textContent = 'بروزرسانی';
                        alert('ارتباط برقرار نشد.');
                    });
                }, {once:false});
            }
        }

        // اجرا وقتی ویجت دیده شد
        function run(){
            fetchData()
              .then(render)
              .catch(function(){
                  var pBox  = document.getElementById('vps7_plugins_box');
                  var tBox  = document.getElementById('vps7_themes_box');
                  pBox.classList.remove('vps7-skel');
                  tBox.classList.remove('vps7-skel');
                  pBox.innerHTML = '<p>عدم دریافت اطلاعات.</p>';
                  tBox.innerHTML = '<p>عدم دریافت اطلاعات.</p>';
              });
        }
        if('IntersectionObserver' in window && cont){
            var io = new IntersectionObserver(function(entries){
                entries.forEach(function(e){
                    if(e.isIntersecting){
                        io.disconnect();
                        // بلافاصله اجرا؛ بدون idle و بدون کش
                        run();
                    }
                });
            }, {rootMargin: '0px 0px 200px 0px'});
            io.observe(cont);
        } else {
            window.addEventListener('load', run);
        }
    })();
    </script>
    <?php
}
