<?php
/**
 * Plugin Name: کلاینت مخزن VPS 7
 * Description: نمایش و نصب افزونه/قالب از سرور، مدیریت لایسنس و بررسی دسترسی (اختصاصی VPS7.net)
 * Author: VPS7.net
 * Version: 1.4.5
 * Text Domain: vps7
 */
if (!defined('ABSPATH')) exit;

if (!class_exists('VPS7_Repository_Client')):

class VPS7_Repository_Client {
    /* تنظیمات ثابت */
    const SERVER_BASE = 'https://vps7.net';
    const API_BASE    = '/wp-json/vps7/v1';

    const VERSION     = '1.4.1';
    const OPT_LICENSE = 'vps7_client_license';

    /* پیام خطای آخرین درخواست دانلود */
    private $last_error = '';

    public function __construct() {
        add_action('admin_menu', array($this, 'admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'admin_assets'));
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'plugin_quick_links'));

        // AJAX
        add_action('wp_ajax_vps7_check_license', array($this, 'ajax_check_license'));
        add_action('wp_ajax_vps7_install_item', array($this, 'ajax_install_item'));
        add_action('wp_ajax_vps7_activate_item', array($this, 'ajax_activate_item'));
        add_action('wp_ajax_vps7_remove_license', array($this, 'ajax_remove_license'));
    }

    private function settings() {
        return array('server_base'=>self::SERVER_BASE, 'api_base'=>self::API_BASE);
    }

    /* -------- گزارش رخدادها به سرور -------- */
    private function env_meta(){
        global $wp_version;
        $curl = function_exists('curl_version') ? curl_version() : array();
        return array(
            'php'         => PHP_VERSION,
            'wp'          => $wp_version ?? '',
            'server'      => isset($_SERVER['SERVER_SOFTWARE']) ? $_SERVER['SERVER_SOFTWARE'] : '',
            'fs_method'   => defined('FS_METHOD') ? FS_METHOD : '',
            'curl'        => isset($curl['version']) ? $curl['version'] : '',
            'ssl'         => isset($curl['ssl_version']) ? $curl['ssl_version'] : '',
            'site'        => site_url(),
            'domain'      => parse_url(site_url(), PHP_URL_HOST),
            'client_ver'  => self::VERSION,
            'plugin_file' => plugin_basename(__FILE__),
        );
    }
    /** ارسال رخداد (fire-and-forget) */
    private function report_event($event, $payload=array(), $level='info'){
        try{
            $s = $this->settings();
            $endpoint = rtrim($s['server_base'], '/').rtrim($s['api_base'], '/').'/client-report';
            if (strpos($endpoint, '/client-report') === false) return;

            $license = get_option(self::OPT_LICENSE, '');
            $body = array(
                'event'   => (string)$event,
                'level'   => (string)$level,
                'license' => $license,
                'domain'  => parse_url(site_url(), PHP_URL_HOST),
                'meta'    => wp_json_encode($this->env_meta()),
                'data'    => wp_json_encode(is_array($payload) ? $payload : array('message'=>(string)$payload)),
            );
            wp_remote_post($endpoint, array('timeout'=>8,'blocking'=>false,'body'=>$body));
        }catch(\Throwable $e){}
    }

    /** IP واقعی کلاینت چه پشت کلادفلر چه مستقیم */
    private function client_ip() {
        foreach (array('HTTP_CF_CONNECTING_IP','HTTP_X_REAL_IP','HTTP_X_FORWARDED_FOR','REMOTE_ADDR') as $k) {
            if (!empty($_SERVER[$k])) {
                $v = $_SERVER[$k];
                if ($k === 'HTTP_X_FORWARDED_FOR') $v = explode(',', $v)[0];
                $ip = is_array($v) ? $v[0] : $v;
                if (preg_match('~^::ffff:(\d{1,3}(?:\.\d{1,3}){3})$~i', $ip, $m)) return $m[1];
                return trim($ip);
            }
        }
        return '0.0.0.0';
    }

    /**
     * چک دسترسی برای UI — نسخه IP-اول (سخت‌گیرانه، بدون کش)
     * ۱) اول بدون لایسنس یک پروب IP می‌زنیم (require_ip=1 + ip واقعی + nocache)
     *    اگر allowed=true و via=ip بود، دسترسی OK.
     * ۲) اگر IP رد شد، و لایسنس داریم، لایسنس را چک می‌کنیم.
     */

	/** اطلاعات کامل دسترسی برای UI (allowed + via) */
public function access_info() {
    $s       = $this->settings();
    $license = trim( get_option(self::OPT_LICENSE, '') );
    $domain  = parse_url(site_url(), PHP_URL_HOST);
    $cip     = $this->client_ip();

    // (1) IP-first probe
$probe = $this->remote_get(
    $s['server_base'].$s['api_base'].'/access',
    array(
        'intent'     => 'connect',
        'consume'    => 0,
        'require_ip' => 1,
        // 'ip'       => $cip, // حذف شد: سرور خودش IP را می‌خواند
        'domain'     => $domain,
        'nocache'    => uniqid('', true),
    )
);
    $via_probe = is_array($probe) && isset($probe['via']) ? strtolower((string)$probe['via']) : '';
    if (is_array($probe) && !empty($probe['allowed']) && $via_probe === 'ip') {
        return array('allowed'=>true, 'via'=>'ip');
    }

    // (2) License check only if IP not allowed
    if ($license === '') return array('allowed'=>false, 'via'=>'ip');

    $acc = $this->remote_get(
        $s['server_base'].$s['api_base'].'/access',
        array(
            'license' => $license,
            'domain'  => $domain,
            'intent'  => 'connect',
            'consume' => 0,
            'nocache' => uniqid('', true),
        )
    );

    if (is_array($acc) && ( !empty($acc['fatal']) || !empty($acc['domain_unbound']) || !empty($acc['invalidate']) || !empty($acc['should_unset']) )) {
        delete_option(self::OPT_LICENSE);
        return array('allowed'=>false, 'via'=>'license');
    }

    if (is_array($acc) && !empty($acc['allowed'])) {
        return array('allowed'=>true, 'via'=>'license');
    }

    return array('allowed'=>false, 'via'=>'license');
}

/** سازگاری قبلی: فقط مقدار بولی */
public function is_allowed_access() {
    $ai = $this->access_info();
    return !empty($ai['allowed']);
}

	
    /* ---------------- Admin UI ---------------- */
    public function admin_menu() {
        $cap = 'install_plugins';
        add_menu_page('مخزن VPS 7', 'مخزن VPS 7', $cap, 'vps7_client', array($this, 'page_plugins'), 'dashicons-cloud', 57);

        add_submenu_page('vps7_client', 'دانلود پلاگین', 'دانلود پلاگین', $cap, 'vps7_client', array($this, 'page_plugins'));
        add_submenu_page('vps7_client', 'دانلود قالب',   'دانلود قالب',   $cap, 'vps7_themes',  array($this, 'page_themes'));

        add_submenu_page('vps7_client', 'تنظیمات API / لایسنس', 'تنظیمات API / لایسنس', $cap, 'vps7_settings', array($this, 'page_settings_include'));
    }
    public function plugin_quick_links($links) { return $links; }

    public function page_settings_include() {
        $file = plugin_dir_path(__FILE__) . 'api-setting.php';
        if (file_exists($file)) {
            require_once $file;
            if (function_exists('vps7_render_api_settings')) {
                vps7_render_api_settings($this);
                return;
            }
        }
        echo '<div class="wrap"><h1 style="font-family:Vazirmatn,sans-serif">تنظیمات</h1><p>فایل api-setting.php یافت نشد.</p></div>';
    }

    /** تبليغات (از سرور) */
    private function fetch_ads(){
        $s = $this->settings();
        $ads = $this->remote_get($s['server_base'].$s['api_base'].'/ads', array());
        if (!is_array($ads)) $ads = array();
        $ads = array_merge(array(
            'plugin_header'=>'','plugin_footer'=>'',
            'theme_header'=>'','theme_footer'=>'',
        ), $ads);
        return $ads;
    }

    public function admin_assets($hook) {
        if (strpos($hook, 'vps7_') === false) return;

        // Font: Vazirmatn (وزیر)
        wp_enqueue_style('vps7-font', 'https://fonts.googleapis.com/css2?family=Vazirmatn:wght@400;600;700;800&display=swap', array(), null);

        // CSS
        wp_register_style('vps7-inline', false, array(), self::VERSION);
        wp_enqueue_style('vps7-inline');
        $css = <<<'CSS'
.wrap{font-family:Vazirmatn,sans-serif; direction:rtl;}
.wrap h1,.wrap h2,.wrap h3,.vps7-theme-title{font-family:Vazirmatn,sans-serif}
.vps7-page .notice,.vps7-page .update-nag{display:none!important}

/* هشدار نبود لایسنس/IP */
.vps7-noaccess{margin:10px 0 14px 0; padding:10px 12px; border-radius:8px; background:#fee2e2; color:#991b1b; font-weight:700}
.vps7-noaccess a{color:#7f1d1d; text-decoration:underline}

/* لیست‌ها */
.vps7-list{display:grid;gap:16px}
.vps7-list--plugin{grid-template-columns:repeat(3, minmax(0,1fr))}
.vps7-list--theme{grid-template-columns:repeat(4, minmax(0,1fr))}
@media (max-width:1450px){ .vps7-list--plugin{grid-template-columns:repeat(2, minmax(0,1fr))} }
@media (max-width:1400px){ .vps7-list--theme{grid-template-columns:repeat(3, minmax(0,1fr))} }
@media (max-width:1100px){ .vps7-list--theme{grid-template-columns:repeat(2, minmax(0,1fr))} }
@media (max-width:700px){ .vps7-list--plugin, .vps7-list--theme{grid-template-columns:1fr} }

/* کارت‌ها */
.vps7-card{display:flex;gap:16px;border:1px solid #e2e8f0;border-radius:10px;padding:12px;background:#fff}
.vps7-card h3{margin:0 0 6px 0;font-size:15px}
.vps7-excerpt{text-align:justify}
.vps7-thumbcol{display:flex; flex-direction:column; align-items:center; gap:8px; min-width:148px}
.vps7-thumbcol img{width:128px;height:128px;border-radius:12px;object-fit:cover}
.vps7-underimg{font-size:12px}
.vps7-underimg a{ text-decoration:none }
.vps7-actions{margin-right:auto;display:flex;flex-direction:column;gap:6px;align-items:stretch;width:188px}
.vps7-cta{display:flex;flex-direction:column;gap:6px}
.vps7-actions .button.vps7-install, .vps7-actions .button.vps7-activate{width:100%; text-align:center}

/* برچسب‌ها */
.vps7-chips{display:flex; flex-wrap:wrap; gap:8px; margin-top:8px}
.vps7-pill{display:inline-block; padding:4px 10px; border-radius:3px; font-size:12px; font-weight:700; color:#fff; text-decoration:none}
.vps7-pill--green{background:#16a34a}
.vps7-pill--red{background:#dc2626}
.vps7-pill--blue{background:#2563eb}

/* ابزارک‌ها */
.vps7-tabs{display:flex;gap:8px;flex-wrap:wrap;margin:10px 0;align-items:center; position:relative; z-index:2}
.vps7-tabs a{padding:6px 10px;border:1px solid #ddd;border-radius:5px;background:#f7fafc;text-decoration:none}
.vps7-sortbar{display:flex;align-items:center;gap:8px;margin:8px 0; position:relative; z-index:1}
.vps7-search{margin:10px 0}

/* قالب‌ها */
.vps7-theme{border:1px solid #e2e8f0;border-radius:10px;background:#fff;overflow:hidden;display:flex;flex-direction:column}
.vps7-theme img{width:100%;height:300px;object-fit:cover}
.vps7-theme .meta{display:flex;gap:8px;flex-wrap:wrap; padding:10px; align-items:center}
.vps7-theme .split{display:grid;grid-template-columns:1fr 1fr 1fr}
.vps7-theme .split a{display:flex;align-items:center;justify-content:center;height:44px;text-decoration:none;font-weight:700}
.vps7-theme .split .right{background:#ef4444;color:#fff}
.vps7-theme .split .middle{background:#ff7a00;color:#fff}
.vps7-theme .split .left{background:#0f172a;color:#fff}
.vps7-theme-title{ text-align:center; font-size:25px; font-weight:800; line-height:0.7; margin:10px 12px 0 12px; }

/* صفحه‌بندی */
.vps7-pager{display:flex;gap:6px;align-items:center;justify-content:center;margin:18px 0}
.vps7-pager .button{min-width:36px;text-align:center}
.vps7-pager .is-active{box-shadow:inset 0 0 0 1px #2271b1}

/* «سایر دسته‌ها» – dropdown سفارشی */
.vps7-catbox{position:relative; width:260px; z-index:9999}
#vps7_cat_input{width:100%}
.vps7-cat-dd{
  position:absolute; top:100%; right:0; width:100%;
  background:#fff; border:1px solid #ddd; border-radius:6px;
  box-shadow:0 8px 20px rgba(0,0,0,.08);
  max-height:320px; overflow:auto; display:none; margin-top:4px;
}
.vps7-cat-dd__item{ padding:8px 10px; cursor:pointer; white-space:nowrap; overflow:hidden; text-overflow:ellipsis; }
.vps7-cat-dd__item:hover{ background:#f3f4f6 }

/* تبلیغات */
.vps7-ads{margin:14px 0}
.vps7-ads .block{background:#fff;border:1px solid #e5e7eb;border-radius:8px;padding:10px}
.vps7-ads img {border-radius:10px;margin-left:10px}

/* — حالت disabled: فقط کم‌رنگ شود، مخفی نشود — */
.button.vps7-install[disabled]{opacity:.45; cursor:not-allowed;}

/* ===== باکس شیک «لایسنس نامعتبر» زیر عنوان ===== */
/* بنر وضعیت دسترسی (سبک یکسان با صفحه بروزرسانی) */
.vps7-infobox{display:flex;gap:12px;align-items:flex-start;border:1px solid #e5e7eb;background:#fff;border-radius:10px;padding:12px 14px;margin:10px 0 14px 0}
.vps7-ok{border-color:#22c55e;background:#ecfdf5}
.vps7-ico{flex:0 0 28px;width:28px;height:28px;border-radius:50%;display:flex;align-items:center;justify-content:center;margin-top:2px;background:#22c55e;color:#fff}
.vps7-infobox h3{margin:0 0 6px 0;font-size:15px}
.vps7-infobox p{margin:0}

/* ===== باکس شیک «لایسنس نامعتبر» زیر عنوان ===== */
.vps7-lic-bad{
  margin:12px 0 16px; padding:14px 16px; border-radius:12px;
  background:linear-gradient(135deg,#fff1f2,#ffe4e6);
  border:1px solid #fecdd3; color:#9f1239;
  display:flex; gap:14px; align-items:flex-start; box-shadow:0 8px 24px rgba(244,63,94,.12);
}
.vps7-licbad-icon{ color:#e11d48; background:#ffe4e6; border:1px solid #fecdd3; width:38px; height:38px; min-width:38px; display:flex; align-items:center; justify-content:center; border-radius:10px; }
.vps7-licbad-body{ flex:1; }
.vps7-licbad-title{ font-weight:800; font-size:15px; margin-bottom:2px; }
.vps7-licbad-desc{ font-weight:600; }
.vps7-licbad-sub{ margin-top:6px; font-size:12px; opacity:.8 }
.vps7-licbad-actions{ display:flex; gap:8px; align-items:center; flex-wrap:wrap; }
.vps7-licbad-actions .button.button-primary{
  background:linear-gradient(135deg,#f43f5e,#fb7185);
  border:0; box-shadow:0 10px 22px rgba(244,63,94,.25); font-weight:800;
}
.vps7-licbad-actions .button.button-link-delete{
  color:#991b1b; border-color:#fecaca; background:#fff; font-weight:700;
}
CSS;
        wp_add_inline_style('vps7-inline', $css);

        // Script
        wp_register_script('vps7-client', false, array('jquery'), self::VERSION, true);
        wp_enqueue_script('vps7-client');

        /* داده‌های لوکال برای JS */
        $s = $this->settings();
        require_once ABSPATH.'wp-admin/includes/plugin.php';

        // افزونه‌ها
        $plugins_map = array();
        $active = array();
        foreach (get_plugins() as $file=>$data) {
            $folder = sanitize_title(explode('/', $file)[0]);
            $name   = isset($data['Name'])       ? sanitize_title($data['Name'])       : '';
            $td     = isset($data['TextDomain']) ? sanitize_title($data['TextDomain']) : '';
            $ver    = isset($data['Version'])    ? $data['Version'] : '';
            foreach (array_filter(array($folder,$name,$td)) as $k){
                $plugins_map[$k] = $ver;
                $plugins_map[str_replace(array('-','_'),'',$k)] = $ver;
            }
        }
        foreach ((array) get_option('active_plugins', array()) as $file){
            $slug = sanitize_title(explode('/', $file)[0]);
            $active[$slug] = true;
            $active[str_replace(array('-','_'),'',$slug)] = true;
        }

        // قالب‌ها
        $themes_map = array();
        foreach (wp_get_themes() as $ss=>$th) {
            $slugt = sanitize_title($ss);
            $themes_map[$slugt] = $th->get('Version');
            $themes_map[str_replace(array('-','_'),'',$slugt)] = $th->get('Version');
        }
        $active_theme = '';
        $t = wp_get_theme(); if ($t) $active_theme = sanitize_title($t->get_stylesheet());
$__acc = $this->access_info();
        wp_localize_script('vps7-client', 'VPS7', array(
            'nonce' => wp_create_nonce('vps7_nonce'),
            'apiRoot' => $s['server_base'].$s['api_base'],
            'allowed'    => (!empty($__acc['allowed']) ? true : false), // از همان access_info
			'allowedVia' => isset($__acc['via']) ? $__acc['via'] : '',
            'installedPlugins' => $plugins_map,
            'installedThemes'  => $themes_map,
            'activePlugins'    => $active,
            'activeTheme'      => $active_theme,
            'perPage'          => 12,
            'texts' => array(
                'loading'   => 'در حال بارگذاری...',
                'installing'=> 'در حال نصب...',
                'activating'=> 'در حال فعال‌سازی...',
                'notFound'  => 'موردی یافت نشد.',
                'noAccess'  => 'دسترسی شما مجاز نیست'
            )
        ));

        $js = <<<'JS'
/* ——— JS اصلی ——— */
(function($){
    var nonce = VPS7.nonce, API = VPS7.apiRoot, PER = VPS7.perPage;

    var state = { type:'plugin', search:'', catSlug:'', catId:'', sort:'latest', page:1, pages:1 };

    function vnorm(s){ return (s||'').toString().toLowerCase().replace(/[\s\-_]+/g,'').trim(); }
    function cmpVer(a,b){var pa=(a||'').toString().split('.').map(Number),pb=(b||'').toString().split('.').map(Number);for(var i=0;i<Math.max(pa.length,pb.length);i++){var x=pa[i]||0,y=pb[i]||0;if(x>y)return 1;if(x<y)return -1;}return 0;}
    function findPlg(slug,title){ var M=VPS7.installedPlugins||{},keys=[]; if(slug){keys.push(slug, slug.replace(/[\s\-_]+/g,''));} if(title){var t=title.toString().toLowerCase(); keys.push(t.replace(/\s+/g,'-'), t.replace(/[\s\-_]+/g,''));} for(var i=0;i<keys.length;i++){ if(M[keys[i]]) return M[keys[i]]; } return ''; }
    function findThm(slug,title){ var M=VPS7.installedThemes||{},keys=[]; if(slug){keys.push(slug, slug.replace(/[\s\-_]+/g,''));} if(title){var t=title.toString().toLowerCase(); keys.push(t.replace(/\s+/g,'-'), t.replace(/[\s\-_]+/g,''));} for(var i=0;i<keys.length;i++){ if(M[keys[i]]) return M[keys[i]]; } return ''; }
    function g2j(gy,gm,gd){var gdm=[31,28,31,30,31,30,31,31,30,31,30,31],jdm=[31,31,31,31,31,31,30,30,30,30,30,29];gy-=1600;gm-=1;gd-=1;var gdn=365*gy+Math.floor((gy+3)/4)-Math.floor((gy+99)/100)+Math.floor((gy+399)/400);for(var i=0;i<gm;++i)gdn+=gdm[i];if(gm>1&&((gy+1600)%4==0&&(((gy+1600)%100!=0)||((gy+1600)%400==0))))gdn++;gdn+=gd;var jdn=gdn-79,jnp=Math.floor(jdn/12053);jdn%=12053;var jy=979+33*jnp+4*Math.floor(jdn/1461);jdn%=1461;if(jdn>=366){jy+=Math.floor((jdn-1)/365);jdn=(jdn-1)%365;}for(var j=0;j<11&&jdn>=jdm[j];++j)jdn-=jdm[j];return [jy,j+1,jdn+1];}
    function pad(n){return (n<10?'0':'')+n;}
    function toJalali(d){ if(!d)return''; if(/^\d{4}-\d{2}-\d{2}/.test(d)){var p=d.split('T')[0].split('-');var j=g2j(+p[0],+p[1],+p[2]);return j[0]+'/'+pad(j[1])+'/'+pad(j[2]);} return d;}

    function renderPager(){
        var c = $('#vps7_pagination'); c.empty(); if(state.pages<=1)return;
        var h='<div class="vps7-pager">';
        h+='<button class="button vps7-pg-prev"'+(state.page<=1?' disabled':'')+'>قبلی</button>';
        var maxBtns=7,start=Math.max(1,state.page-3),end=Math.min(state.pages,start+maxBtns-1);
        if(end-start+1<maxBtns)start=Math.max(1,end-maxBtns+1);
        for(var i=start;i<=end;i++){ h+='<button class="button vps7-pg-num'+(i===state.page?' is-active':'')+'" data-page="'+i+'">'+i+'</button>'; }
        h+='<button class="button vps7-pg-next"'+(state.page>=state.pages?' disabled':'')+'>بعدی</button></div>';
        c.html(h);
    }
    function buildParams(){ var p={search:state.search||'',per_page:PER,page:state.page}; if(state.catSlug){p.category=state.catSlug;p.category_slug=state.catSlug;p.slug=state.catSlug;p.term=state.catSlug;} if(state.catId){p.category_id=state.catId;} if(state.sort){p.sort=state.sort;} return p; }
    function fetchList(){
        var url = API + (state.type==='theme'?'/themes':'/plugins');
        $('#vps7_list').html('<p>'+(VPS7.texts?VPS7.texts.loading:'در حال بارگذاری...')+'</p>');
        $.get(url, buildParams(), function(res){
            var items=(res&&res.items)?res.items:[]; state.pages=(res&&res.pages)?parseInt(res.pages,10):1;
            if(state.type==='theme') renderThemes(items); else renderPlugins(items);
            renderPager();

            if(!VPS7.allowed){ $('.vps7-install').prop('disabled', true).attr('title', VPS7.texts?VPS7.texts.noAccess:''); }
        });
    }

    function renderPlugins(arr){
        var html='';
        if(arr&&arr.length){
            arr.forEach(function(it){
                var excerptFull=(it.excerpt||'').replace(/<[^>]*>/g,'');
                var slug=(it.slug||'')||((it.title||'').toString().toLowerCase().replace(/\s+/g,'-'));
                var inst=findPlg(slug,it.title||'');
                var label='نصب', dis='', titleAttr='';
                if(inst){ if(it.version && cmpVer(it.version,inst)<=0){ label='به‌روز هستید'; dis='disabled'; } else { label='بروزرسانی'; } }
                if(!VPS7.allowed){ dis='disabled'; titleAttr=' title="'+(VPS7.texts?VPS7.texts.noAccess:'')+'" '; }
                var isActive=false, nslug=vnorm(slug); if(inst && VPS7.activePlugins){ if(VPS7.activePlugins[slug]||VPS7.activePlugins[nslug]) isActive=true; }
                var actDis=isActive?' disabled title="در حال حاضر فعال است"':'';
                var udate=toJalali(it.updated_at||'');
                html+='<div class="vps7-card" data-id="'+it.id+'" data-version="'+(it.version||'')+'" data-title="'+(it.title||'')+'" data-slug="'+slug+'">'
                    +'<div class="vps7-thumbcol"><img src="'+(it.thumbnail||'')+'"><div class="vps7-underimg">'
                    +'<a href="#" class="button-link vps7-details"'
                    +' data-full="'+((it.content||'').replace(/"/g,'&quot;'))+'"'
                    +' data-title="'+(it.title||'')+'"'
                    +' data-img="'+(it.thumbnail||'')+'"'
                    +' data-ver="'+(it.version||'')+'"'
                    +' data-udate="'+udate+'"'
                    +' data-dl="'+(it.downloads||0)+'"'
                    +' data-excerpt="'+excerptFull.replace(/"/g,'&quot;')+'">جزئیات</a>'
                    +(it.link?' | <a href="'+it.link+'" target="_blank" class="button-link">صفحه افزونه</a>':'')
                    +'</div></div>'
                    +'<div><h3>'+(it.title||'')+'</h3>'
                    +'<div class="vps7-excerpt">'+(excerptFull.length>150?excerptFull.slice(0,150)+'…':excerptFull)+'</div>'
                    +'<div class="vps7-chips"><a class="button vps7-pill vps7-pill--green">نسخه '+(it.version||'')+'</a> <a class="button vps7-pill vps7-pill--red">بروزرسانی '+udate+'</a></div>'
                    +'</div>'
                    +'<div class="vps7-actions">'
                    +   '<div class="vps7-cta">'
                    +     '<button class="button button-primary vps7-install" '+titleAttr+' '+dis+' data-type="plugin">'+label+'</button>'
                    +     (inst?('<button class="button vps7-activate" '+actDis+' data-type="plugin">فعال‌سازی</button>'):'')
                    +   '</div>'
                    +   '<a class="button vps7-pill vps7-pill--blue">دانلود '+(it.downloads||0)+'</a>'
                    +'</div>'
                +'</div>';
            });
        } else { html='<p>'+(VPS7.texts?VPS7.texts.notFound:'موردی یافت نشد.')+'</p>'; }
        $('#vps7_list').html(html);
    }
    function renderThemes(arr){
        var html='';
        if(arr&&arr.length){
            arr.forEach(function(it){
                var slug=(it.slug||'')||((it.title||'').toString().toLowerCase().replace(/\s+/g,'-'));
                var inst=findThm(slug,it.title||'');
                var label='نصب', dis=''; if(inst){ var c=cmpVer(it.version,inst); if(it.version && c<=0){ label='به‌روز هستید'; dis='disabled'; } else { label='بروزرسانی'; } }
                if(!VPS7.allowed){ dis='disabled'; }
                var isActive=(VPS7.activeTheme && (VPS7.activeTheme===slug || vnorm(VPS7.activeTheme)===vnorm(slug))) ? ' disabled title="در حال حاضر فعال است"' : '';
                var previewAttr = it.preview_url ? (' href="'+it.preview_url+'" target="_blank" ') : '';
                var udate=toJalali(it.updated_at||'');
                html+='<div class="vps7-theme" data-id="'+it.id+'" data-version="'+(it.version||'')+'" data-title="'+(it.title||'')+'" data-slug="'+slug+'">'
                    +  '<img src="'+(it.thumbnail||'')+'" alt="">'
                    +  '<h3 class="vps7-theme-title">'+(it.title||'')+'</h3>'
                    +  '<div class="meta">'
                    +     '<a class="button vps7-pill vps7-pill--green">نسخه '+(it.version||'')+'</a>'
                    +     '<a class="button vps7-pill vps7-pill--red">بروزرسانی '+udate+'</a>'
                    +     '<a class="button vps7-pill vps7-pill--blue">دانلود '+(it.downloads||0)+'</a>'
                    +     '<button class="button button-primary vps7-install" '+dis+' data-type="theme">'+label+'</button>'
                    +     (inst?('<button class="button vps7-activate" '+isActive+' data-type="theme">فعال‌سازی</button>'):'')
                    +  '</div>'
                    +  '<div class="split">'
                    +     '<a href="#" class="right vps7-details"'
                    +' data-full="'+((it.content||'').replace(/"/g,'&quot;'))+'"'
                    +' data-title="'+(it.title||'')+'"'
                    +' data-img="'+(it.thumbnail||'')+'"'
                    +' data-ver="'+(it.version||'')+'"'
                    +' data-udate="'+udate+'"'
                    +' data-dl="'+(it.downloads||0)+'"'
                    +' data-excerpt="'+( (it.excerpt||'').replace(/<[^>]*>/g,'').replace(/"/g,'&quot;') )+'">جزئیات قالب</a>'
                    +     ( it.preview_url ? ('<a class="middle" '+previewAttr+'>پیش‌نمایش قالب</a>')
                                         : ('<a href="#" class="middle vps7-preview" data-img="'+(it.thumbnail||'')+'" data-title="'+(it.title||'')+'">پیش‌نمایش قالب</a>') )
                    +     (it.link?('<a class="left" href="'+it.link+'" target="_blank">صفحه قالب</a>'):'<span class="left" style="opacity:.6;cursor:not-allowed">صفحه قالب</span>')
                    +  '</div>'
                    +'</div>';
            });
        } else { html='<p>'+(VPS7.texts?VPS7.texts.notFound:'موردی یافت نشد.')+'</p>'; }
        $('#vps7_list').html(html);
    }

    /* حذف لایسنس ذخیره‌شده از باکس هشدار */
    $(document).on('click', '#vps7_remove_license_btn', function(e){
      e.preventDefault();
      var $btn = $(this);
      if(!confirm('لایسنس ذخیره‌شده حذف شود؟')) return;
      $btn.prop('disabled', true);
      $.post(ajaxurl, { action:'vps7_remove_license', nonce: VPS7.nonce }, function(res){
        location.reload();
      }).fail(function(){
        alert('خطا در ارتباط');
        $btn.prop('disabled', false);
      });
    });

    /* تب‌ها و فیلترها و ... */
    $(document).on('click','.vps7-tab', function(e){
        e.preventDefault();
        $('.vps7-tab').removeClass('active'); $(this).addClass('active');
        state.catSlug = $(this).data('cat') || '';
        state.catId   = $(this).data('id')  || '';
        state.page    = 1;
        fetchList();
    });

    /* سایر دسته‌ها – dropdown سفارشی */
    var ALL = window.VPS7_ALL_CATS || [];
    function faNorm(s){ return (s||'').toString().replace(/[ي]/g,'ی').replace(/[ك]/g,'ک').replace(/[ۀة]/g,'ه').replace(/\u200c/g,' ').toLowerCase().trim(); }
    function filterCats(q){
      var f = faNorm(q||''); if(!f) return ALL.slice(0,200);
      return ALL.filter(function(c){
        var n = faNorm(c.name||''), s = faNorm(c.slug||'');
        return n.indexOf(f)!==-1 || s.indexOf(f)!==-1;
      }).slice(0,200);
    }
    function renderCatDD(list){
      var box = $('#vps7_cat_dd'); box.empty();
      if(!list || !list.length){ box.hide(); return; }
      var html=''; list.forEach(function(c){ html+='<div class="vps7-cat-dd__item" data-slug="'+(c.slug||'')+'" data-id="'+(c.id||'')+'">'+(c.name||c.slug||'')+'</div>'; });
      box.html(html).show();
    }
    $(document).on('focus input', '#vps7_cat_input', function(){ renderCatDD(filterCats($(this).val())); });
    $(document).on('click', '.vps7-cat-dd__item', function(){
      var slug=$(this).data('slug')||'', id=$(this).data('id')||'';
      $('#vps7_cat_input').val($(this).text()); $('#vps7_cat_dd').hide();
      state.catSlug=slug; state.catId=id; state.page=1; fetchList();
    });
    $(document).on('click', function(e){ if(!$(e.target).closest('.vps7-catbox').length){ $('#vps7_cat_dd').hide(); } });

    /* مرتب‌سازی */
    $('#vps7_sort').on('change', function(){ state.sort = $(this).val() || 'latest'; state.page = 1; fetchList(); });

    /* جستجو */
    $('#vps7_search_btn').on('click', function(){ state.search = $('#vps7_search').val(); state.page = 1; fetchList(); });

    /* صفحه‌بندی */
    $(document).on('click','.vps7-pg-num', function(){ var p=parseInt($(this).data('page'),10)||1; if(p!==state.page){ state.page=p; fetchList(); } });
    $(document).on('click','.vps7-pg-prev', function(){ if(state.page>1){ state.page--; fetchList(); } });
    $(document).on('click','.vps7-pg-next', function(){ if(state.page<state.pages){ state.page++; fetchList(); } });

    /* نصب و فعال‌سازی */
    $(document).on('click','.vps7-install', function(e){
      e.preventDefault();
      var card=$(this).closest('[data-id]'); 
      var id=card.data('id'); 
      var type=$(this).data('type');
      var btn=$(this); 
      btn.prop('disabled',true).text(VPS7.texts?VPS7.texts.installing:'در حال نصب...');

      $.post(ajaxurl, { action:'vps7_install_item', nonce:nonce, id:id, type:type }, function(res){
        if(res && res.success){
          if(type==='theme'){
            var act = $('<button class="button vps7-activate" data-type="theme">فعال‌سازی</button>');
            btn.replaceWith(act);
          }else{
            btn.text('نصب شد').prop('disabled',true);
            if(!card.find('.vps7-activate').length){
              card.find('.vps7-cta').append('<button class="button vps7-activate" data-type="plugin">فعال‌سازی</button>');
            }
          }
        } else {
          alert('خطا: '+(res && res.data ? res.data : 'نامشخص'));
          if(VPS7.allowed) btn.prop('disabled',false).text('نصب');
          else btn.prop('disabled',true).text('نصب');
        }
      }).fail(function(){
        alert('ارتباط برقرار نشد.');
        if(VPS7.allowed) btn.prop('disabled',false).text('نصب');
        else btn.prop('disabled',true).text('نصب');
      });
    });

    $(document).on('click','.vps7-activate', function(e){
        e.preventDefault();
        var card=$(this).closest('[data-id]');
        var title=card.data('title');
        var slug =card.data('slug');
        var type=$(this).data('type');
        var btn=$(this);
        btn.prop('disabled',true).text(VPS7.texts?VPS7.texts.activating:'در حال فعال‌سازی...');
        $.post(ajaxurl, { action:'vps7_activate_item', nonce:nonce, type:type, title:title, slug:slug }, function(res){
            if(res && res.success){
              btn.text('فعال شد').prop('disabled',true).attr('title','در حال حاضر فعال است');
            } else {
              alert(res && res.data ? res.data : 'خطا در فعال‌سازی');
              btn.prop('disabled',false).text('فعال‌سازی');
            }
        }).fail(function(){
            alert('ارتباط برقرار نشد.');
            btn.prop('disabled',false).text('فعال‌سازی');
        });
    });

    $(function(){
        var list = $('#vps7_list');
        if(list.length){ state.type = list.data('type') || 'plugin'; }

        if(!VPS7.allowed){
            $('.vps7-install').prop('disabled', true).attr('title', VPS7.texts?VPS7.texts.noAccess:'');
        }

        fetchList();
    });
})(jQuery);
JS;
        wp_add_inline_script('vps7-client', $js);
    }

    /* ---------- License invalid box (HTML) ---------- */
    private function render_license_invalid_box_html() {
        $code = trim(get_option(self::OPT_LICENSE, ''));
        if ($code === '') return '';

        $s    = $this->settings();
        $info = $this->remote_get($s['server_base'].$s['api_base'].'/license-info', array(
            'license' => $code,
            'nocache' => uniqid('', true),
        ));

        $invalid  = false;
        $reason   = 'لایسنس این سایت معتبر نیست.';
        $ends_at  = '';
        $status   = '';

        if (is_array($info) && !empty($info['ok']) && !empty($info['license'])) {
            $lic    = $info['license'];
            $status = isset($lic['status']) ? (string)$lic['status'] : '';
            $end_at = isset($lic['end_at']) ? (string)$lic['end_at'] : '';

            $expired = false;
            if ($end_at) {
                $ts = strtotime($end_at);
                if ($ts && $ts < time()) $expired = true;
            }
            if ($status !== 'active' || $expired) {
                $invalid = true;
                if ($expired) {
                    $reason  = 'اشتراک لایسنس شما منقضی شده است.';
                    $ends_at = $end_at;
                } else {
                    $reason = 'وضعیت لایسنس غیر‌فعال/لغوشده است.';
                }
            }
        } else {
            $invalid = true;
            $reason  = 'لایسنس پیدا نشد یا معتبر نیست.';
        }

        if (!$invalid) return '';

        $shop_link     = esc_url('https://vps7.net/shop');
        $settings_link = esc_url( admin_url('admin.php?page=vps7_settings') );
        $ends_html     = $ends_at ? '<div class="vps7-licbad-sub">تاریخ پایان: <code>'.esc_html($ends_at).'</code></div>' : '';

        return '
<div class="vps7-lic-bad">
  <div class="vps7-licbad-icon" aria-hidden="true">
    <svg viewBox="0 0 24 24" width="22" height="22" fill="currentColor"><path d="M11.001 10h2v5h-2zm0 6h2v2h-2z"/><path d="M1 21h22L12 2 1 21z"/></svg>
  </div>
  <div class="vps7-licbad-body">
    <div class="vps7-licbad-title">لایسنس نامعتبر است</div>
    <div class="vps7-licbad-desc">'.esc_html($reason).'</div>
    '.$ends_html.'
  </div>
  <div class="vps7-licbad-actions">
    <a class="button button-primary" href="'.$shop_link.'" target="_blank" rel="noopener">خرید / تمدید</a>
    <a class="button" href="'.$settings_link.'">بخش لایسنس</a>
    <button class="button button-link-delete" id="vps7_remove_license_btn" title="حذف لایسنس ذخیره‌شده">حذف لایسنس</button>
  </div>
</div>';
    }

    /* ---------- Helpers ---------- */
    public function g2j_php($gy,$gm,$gd,$as_string=true){
        $gdm = array(31,28,31,30,31,30,31,31,30,31,30,31);
        $jdm = array(31,31,31,31,31,31,30,30,30,30,30,29);
        $gy2 = $gy - 1600; $gm2 = $gm - 1; $gd2 = $gd - 1;
        $g_day_no = 365*$gy2 + intdiv($gy2+3,4) - intdiv($gy2+99,100) + intdiv($gy2+399,400);
        for ($i=0; $i<$gm2; ++$i) $g_day_no += $gdm[$i];
        if ($gm2>1 && (($gy%4==0 && $gy%100!=0) || ($gy%400==0))) $g_day_no++;
        $g_day_no += $gd2;
        $j_day_no = $g_day_no - 79;
        $j_np = intdiv($j_day_no,12053); $j_day_no %= 12053;
        $jy = 979 + 33*$j_np + 4*intdiv($j_day_no,1461);
        $j_day_no %= 1461;
        if ($j_day_no >= 366) { $jy += intdiv($j_day_no-1,365); $j_day_no = ($j_day_no-1)%365; }
        for ($i=0; $i<11 && $j_day_no>=$jdm[$i]; ++$i) $j_day_no -= $jdm[$i];
        $jm = $i+1; $jd = $j_day_no+1;
        if ($as_string) return sprintf('%04d/%02d/%02d', $jy,$jm,$jd);
        return array($jy,$jm,$jd);
    }
    public function normalize_updated($d){
        if (!$d) return '';
        if (preg_match('/^\d{4}-\d{2}-\d{2}/', $d)) {
            $parts = explode('T',$d)[0];
            list($gy,$gm,$gd) = array_map('intval', explode('-',$parts));
            return $this->g2j_php($gy,$gm,$gd,true);
        }
        return $d;
    }

    /* ---------------- Pages (دانلودها) ---------------- */
    public function page_plugins() { $this->render_store('plugin'); }
    public function page_themes()  { $this->render_store('theme'); }

    private function render_store($type) {
        $s = $this->settings();
        $allowed = $this->is_allowed_access();

        $cats  = $this->remote_get($s['server_base'].$s['api_base'].'/categories', array('type'=>($type==='theme'?'theme':'plugin')));
        $items = $this->remote_get($s['server_base'].$s['api_base'].'/'.($type==='theme'?'themes':'plugins'), array('page'=>1,'per_page'=>12,'sort'=>'latest'));
        $ads   = $this->fetch_ads();

        $title = ($type==='theme' ? 'دانلود قالب' : 'دانلود پلاگین');

        echo '<div class="wrap vps7-page">';
  echo '<h1 style="font-family:Vazirmatn,sans-serif">'.$title.'</h1>';

// باکس «لایسنس نامعتبر» — دقیقا زیر عنوان
$box = $this->render_license_invalid_box_html();
if ($box) echo $box;

// وضعیت دسترسی برای بنر
$acc = $this->access_info();
$via = isset($acc['via']) ? (string)$acc['via'] : '';

// اگر مجاز بود (چه با IP چه با لایسنس) بنر سبز با تیک نشان بده
if (!empty($acc['allowed']) && ($via === 'ip' || $via === 'license')) {
    $ok_title = ($via === 'ip') ? 'دسترسی IP شما مجاز است' : 'دسترسی با لایسنس معتبر';
    $ok_desc  = ($via === 'ip') ? 'نیازی به ثبت لایسنس نیست.' : 'این دسترسی به‌واسطهٔ لایسنس فعال شده است.';
    echo '<div class="vps7-infobox vps7-ok">
            <div class="vps7-ico">
              <svg viewBox="0 0 24 24" width="18" height="18" fill="currentColor"><path d="M9 16.2l-3.5-3.5L4 14.2l5 5 11-11-1.5-1.5z"/></svg>
            </div>
            <div><h3>'.$ok_title.'</h3><p>'.$ok_desc.'</p></div>
          </div>';
}

// تبلیغ هدر
$hdr = ($type==='theme') ? $ads['theme_header'] : $ads['plugin_header'];
if (!empty($hdr)) {
    echo '<div class="vps7-ads"><div class="block">'.wp_kses_post($hdr).'</div></div>';
}

// پیام «عدم دسترسی»
if (!$allowed) {
    $settings_link = esc_url( admin_url('admin.php?page=vps7_settings') );
    $shop_link     = esc_url( 'https://vps7.net/shop' );
    echo '<div class="vps7-noaccess">';
    echo 'شما اجازه دسترسی به ریپازیتوری VPS 7 ندارید. برای فعال‌سازی دسترسی از طریق لایسنس به ';
    echo '<a href="'.$settings_link.'">بخش لایسنس</a>  مراجعه کنید، ';
    echo 'و برای خرید لایسنس به <a href="'.$shop_link.'" target="_blank" rel="noopener">VPS 7</a> مراجعه کنید.';
    echo '</div>';
}


        if (!is_array($cats)) $cats = array();
        $top = array_slice($cats, 0, 10);

        echo '<div class="vps7-search"><input type="search" id="vps7_search" placeholder="جستجو..." class="regular-text"> <button class="button" id="vps7_search_btn" data-type="'.$type.'">جستجو</button></div>';

        echo '<div class="vps7-tabs">';
        echo '<a href="#" class="vps7-tab active" data-cat="" data-id="">همه '.($type==='theme'?'قالب‌ها':'پلاگین‌ها').'</a>';
        foreach ($top as $c) {
            $slug = isset($c['slug']) ? $c['slug'] : '';
            $id   = isset($c['id'])   ? $c['id']   : '';
            echo '<a href="#" class="vps7-tab" data-cat="'.esc_attr($slug).'" data-id="'.esc_attr($id).'">'.esc_html($c['name']).'</a>';
        }
        if (count($cats) > 10) {
            wp_add_inline_script('vps7-client', 'window.VPS7_ALL_CATS = '.wp_json_encode(array_values($cats)).';', 'before');
            echo '<span style="flex:1"></span>';
            echo '<div class="vps7-catbox">';
            echo '<label for="vps7_cat_input" style="display:block;margin-bottom:4px;">سایر دسته‌ها:</label>';
            echo '<input id="vps7_cat_input" class="vps7-select" placeholder="نام دسته..." autocomplete="off" />';
            echo '<div id="vps7_cat_dd" class="vps7-cat-dd"></div>';
            echo '</div>';
        }
        echo '</div>';

        echo '<div class="vps7-sortbar"><label for="vps7_sort">مرتب‌سازی:</label> ';
        echo '<select id="vps7_sort"><option value="latest" selected>جدیدترین‌ها</option><option value="popular">محبوب‌ترین (دانلود بیشتر)</option></select></div>';

        $listClass = $type==='theme' ? 'vps7-list vps7-list--theme' : 'vps7-list vps7-list--plugin';
        echo '<div id="vps7_list" class="'.$listClass.'" data-type="'.$type.'">';
        if (!empty($items['items']) && is_array($items['items'])) {
            if ($type==='theme') $this->render_theme_items($items['items'], $allowed);
            else $this->render_plugin_items($items['items'], $allowed);
        } else {
            echo '<p>موردی یافت نشد.</p>';
        }
        echo '</div>';

        // تبلیغ فوتر
        $ftr = ($type==='theme') ? $ads['theme_footer'] : $ads['plugin_footer'];
        if (!empty($ftr)) {
            echo '<div class="vps7-ads"><div class="block">'.wp_kses_post($ftr).'</div></div>';
        }

        echo '<div id="vps7_pagination"></div>';
        echo '</div>';
    }

    private function excerpt150($txt){
        $t = wp_strip_all_tags($txt);
        if (function_exists('mb_substr')) {
            $e = mb_substr($t, 0, 150, 'UTF-8'); $more = (mb_strlen($t, 'UTF-8')>150) ? '…' : '';
        } else {
            $e = substr($t, 0, 150); $more = (strlen($t)>150)?'…':'';
        }
        return $e.$more;
    }

    private function render_plugin_items($items, $allowed) {
        if (empty($items)) { echo '<p>موردی یافت نشد.</p>'; return; }
        require_once ABSPATH.'wp-admin/includes/plugin.php';
        foreach ($items as $it) {
            $img        = esc_url(!empty($it['thumbnail']) ? $it['thumbnail'] : 'https://s.w.org/images/core/emoji/14.0.0/svg/1f4e6.svg');
            $title      = esc_html($it['title']);
            $updated_d  = isset($it['updated_at']) ? $this->normalize_updated($it['updated_at']) : '';
            $ver        = isset($it['version']) ? $it['version'] : '';
            $dl         = isset($it['downloads']) ? intval($it['downloads']) : 0;

            $installed  = false; $local_ver=''; $is_active=false; $plugin_file='';
            $slug = !empty($it['slug']) ? sanitize_title($it['slug']) : sanitize_title($it['title']);
            foreach (get_plugins() as $file=>$data) {
                $folder = explode('/', $file)[0];
                if (sanitize_title($folder) === $slug || sanitize_title($data['Name']) === $slug) {
                    $installed=true; $local_ver=isset($data['Version'])?$data['Version']:''; $plugin_file=$file; break;
                }
            }
            if ($installed && function_exists('is_plugin_active') && $plugin_file) { $is_active = is_plugin_active($plugin_file); }

            $btn_label='نصب'; $btn_disabled=false;
            if ($installed) {
                if ($ver && version_compare($ver, $local_ver, '<=')) { $btn_label='به‌روز هستید'; $btn_disabled=true; }
                else { $btn_label='بروزرسانی'; }
            }
            if (!$allowed) { $btn_disabled = true; }

            $dis_attr   = $btn_disabled ? ' disabled ' : '';
            $title_attr = !$allowed ? ' title="دسترسی شما مجاز نیست"' : '';

            $excerpt_full  = wp_strip_all_tags(isset($it['excerpt']) ? $it['excerpt'] : '');
            $excerpt_short = esc_html($this->excerpt150(isset($it['excerpt'])?$it['excerpt']:''));

            $page_link_html = '';
            if (!empty($it['link'])) {
                $page_link_html = ' | <a href="'.esc_url($it['link']).'" target="_blank" class="button-link">صفحه افزونه</a>';
            }
            $activate_btn_html = '';
            if ($installed) {
                $activate_btn_html = ' <button class="button vps7-activate" '.($is_active?'disabled title="در حال حاضر فعال است"':'').' data-type="plugin">فعال‌سازی</button>';
            }

            $details_attrs = 'data-full="'.esc_attr(isset($it['content'])?$it['content']:'').'" '.
                             'data-title="'.$title.'" '.
                             'data-img="'.$img.'" '.
                             'data-ver="'.esc_attr($ver).'" '.
                             'data-udate="'.esc_attr($updated_d).'" '.
                             'data-dl="'.esc_attr($dl).'" '.
                             'data-excerpt="'.esc_attr($excerpt_full).'"';

            $html = <<<HTML
<div class="vps7-card" data-id="{ID}" data-version="{$ver}" data-title="{$title}" data-slug="{$slug}">
    <div class="vps7-thumbcol">
        <img src="{$img}" alt="">
        <div class="vps7-underimg">
            <a href="#" class="button-link vps7-details" {$details_attrs}>جزئیات</a>{$page_link_html}
        </div>
    </div>
    <div>
        <h3>{$title}</h3>
        <div class="vps7-excerpt">{$excerpt_short}</div>
        <div class="vps7-chips">
            <a class="button vps7-pill vps7-pill--green">نسخه {$ver}</a>
            <a class="button vps7-pill vps7-pill--red">بروزرسانی {$updated_d}</a>
        </div>
    </div>
    <div class="vps7-actions">
        <div class="vps7-cta">
            <button class="button button-primary vps7-install"{$dis_attr}{$title_attr} data-type="plugin">{$btn_label}</button>{$activate_btn_html}
        </div>
        <a class="button vps7-pill vps7-pill--blue">دانلود {$dl}</a>
    </div>
</div>
HTML;
            echo str_replace('{ID}', intval($it['id']), $html);
        }
    }

    private function render_theme_items($items, $allowed) {
        if (empty($items)) { echo '<p>موردی یافت نشد.</p>'; return; }
        require_once ABSPATH.'wp-admin/includes/theme.php';

        foreach ($items as $it) {
            $img        = esc_url(!empty($it['thumbnail']) ? $it['thumbnail'] : 'https://s.w.org/images/core/emoji/14.0.0/svg/1f4f7.svg');
            $title      = esc_html($it['title']);
            $updated_d  = isset($it['updated_at']) ? $this->normalize_updated($it['updated_at']) : '';
            $ver        = isset($it['version']) ? $it['version'] : '';
            $dl         = isset($it['downloads']) ? intval($it['downloads']) : 0;

            $slug = !empty($it['slug']) ? sanitize_title($it['slug']) : sanitize_title($it['title']);

            // نسخه نصب‌شده و فعال بودن
            $installedVer = '';
            foreach (wp_get_themes() as $ss=>$th) { if (sanitize_title($ss)===$slug) { $installedVer=$th->get('Version'); break; } }
            $isActive = false; $t = wp_get_theme(); if ($t) $isActive = (sanitize_title($t->get_stylesheet())===$slug);

            // وضعیت نصب/آپدیت
            $btn_label='نصب'; $btn_disabled=false;
            if ($installedVer) {
                if ($ver && version_compare($ver, $installedVer, '<=')) { $btn_label='به‌روز هستید'; $btn_disabled=true; }
                else { $btn_label='بروزرسانی'; }
            }
            if (!$allowed) { $btn_disabled = true; }

            $dis_attr = $btn_disabled ? ' disabled' : '';
            $preview_url = !empty($it['preview_url']) ? esc_url($it['preview_url']) : '';

            $excerpt_full  = wp_strip_all_tags(isset($it['excerpt']) ? $it['excerpt'] : '');
            $excerpt_short = esc_html($this->excerpt150(isset($it['excerpt'])?$it['excerpt']:''));

            $details_attrs = 'data-full="'.esc_attr(isset($it['content'])?$it['content']:'').'" '.
                             'data-title="'.$title.'" '.
                             'data-img="'.$img.'" '.
                             'data-ver="'.esc_attr($ver).'" '.
                             'data-udate="'.esc_attr($updated_d).'" '.
                             'data-dl="'.esc_attr($dl).'" '.
                             'data-excerpt="'.esc_attr($excerpt_full).'"';

            $preview_html = $preview_url
                ? '<a class="middle" href="'.$preview_url.'" target="_blank">پیش‌نمایش قالب</a>'
                : '<a href="#" class="middle vps7-preview" data-img="'.$img.'" data-title="'.$title.'">پیش‌نمایش قالب</a>';

            $page_link_html = !empty($it['link'])
                ? ' <a class="left" href="'.esc_url($it['link']).'" target="_blank">صفحه قالب</a>'
                : ' <span class="left" style="opacity:.6;cursor:not-allowed">صفحه قالب</span>';

            $activate_btn_html2 = $installedVer
                ? ('<button class="button vps7-activate" '.($isActive ? 'disabled title="در حال حاضر فعال است"' : '').' data-type="theme">فعال‌سازی</button>')
                : '';

            $html = <<<HTML
<div class="vps7-theme" data-id="{ID}" data-version="{$ver}" data-title="{$title}" data-slug="{$slug}">
    <img src="{$img}" alt="">
    <h3 class="vps7-theme-title">{$title}</h3>
    <div class="meta">
        <a class="button vps7-pill vps7-pill--green">نسخه {$ver}</a>
        <a class="button vps7-pill vps7-pill--red">بروزرسانی {$updated_d}</a>
        <a class="button vps7-pill vps7-pill--blue">دانلود {$dl}</a>
        <button class="button button-primary vps7-install"{$dis_attr} data-type="theme">{$btn_label}</button>
        {$activate_btn_html2}
    </div>
    <div class="split">
        <a href="#" class="right vps7-details" {$details_attrs}>جزئیات قالب</a>
        {$preview_html}
        {$page_link_html}
    </div>
</div>
HTML;
            echo str_replace('{ID}', intval($it['id']), $html);
        }
    }

    /* ---------------- AJAX ---------------- */
    public function ajax_check_license() {
        check_ajax_referer('vps7_nonce', 'nonce');
        $license = isset($_POST['license']) ? sanitize_text_field($_POST['license']) : '';
        if (!$license) wp_send_json_error('لطفاً لایسنس را وارد کنید.');
        $domain = parse_url(site_url(), PHP_URL_HOST);
        $s = $this->settings();
        $res = $this->remote_post($s['server_base'].$s['api_base'].'/verify-license', array('license'=>$license,'domain'=>$domain));
        if (is_array($res) && !empty($res['ok'])) { update_option(self::OPT_LICENSE, $license); wp_send_json_success($res['license']); }
        wp_send_json_error('لایسنس نامعتبر است.');
    }

    /** گرفتن لینک دانلود (با اولویت لینک امضاشده) — با bind ایمن */
    private function resolve_download_url($type, $id){
        $s = $this->settings();
        $license = get_option(self::OPT_LICENSE, '');
        $domain  = parse_url(site_url(), PHP_URL_HOST);

        // وقتی لایسنس خالی است، bind=0 تا اتوبایند انجام نشود؛
        $payload = array(
            'type'=>$type,
            'id'=>$id,
            'domain'=>$domain,
            'consume'=>1,
        );
        if ($license !== '') {
            $payload['license'] = $license;
            $payload['bind']    = 1;
        } else {
            $payload['bind']    = 0;
        }

        $resp = $this->remote_get($s['server_base'].$s['api_base'].'/item-download', $payload);

        if (is_array($resp) && (!empty($resp['invalidate']) || !empty($resp['should_unset']))) {
            $this->report_event('item_download_denied', array('type'=>$type,'id'=>$id,'resp'=>$resp), 'warn');
            delete_option(self::OPT_LICENSE);
            $this->last_error = !empty($resp['message']) ? $resp['message'] : (!empty($resp['reason']) ? $resp['reason'] : '');
            return '';
        }

        if (is_array($resp) && !empty($resp['signed_url'])) return esc_url_raw($resp['signed_url']);
        if (is_array($resp) && !empty($resp['download_url'])) return esc_url_raw($resp['download_url']);
        if (is_array($resp) && !empty($resp['url']))          return esc_url_raw($resp['url']);

        if (is_array($resp)) {
            if (!empty($resp['message'])) $this->last_error = $resp['message'];
            elseif (!empty($resp['reason'])) $this->last_error = $resp['reason'];
        }

        $u2 = $this->remote_get($s['server_base'].$s['api_base'].'/'.($type==='theme'?'themes':'plugins'), array('id'=>$id, 'per_page'=>1));
        if (is_array($u2) && !empty($u2['items']) && !empty($u2['items'][0]['download_url'])) return esc_url_raw($u2['items'][0]['download_url']);
        $u3 = $this->remote_get($s['server_base'].$s['api_base'].'/'.($type==='theme'?'theme':'plugin'), array('id'=>$id));
        if (is_array($u3) && !empty($u3['download_url'])) return esc_url_raw($u3['download_url']);

        $this->report_event('resolve_download_url_failed', array('type'=>$type,'id'=>$id,'resp'=>$resp), 'error');
        return '';
    }

    public function ajax_install_item() {
        check_ajax_referer('vps7_nonce', 'nonce');
        if (!current_user_can('install_plugins') && !current_user_can('install_themes')) wp_send_json_error('دسترسی ندارید.');
        if (!$this->is_allowed_access()) { wp_send_json_error('دسترسی شما مجاز نیست'); }

        $type = isset($_POST['type']) ? sanitize_text_field($_POST['type']) : 'plugin';
        $post_id = isset($_POST['id']) ? intval($_POST['id']) : 0;
        if (!$post_id) wp_send_json_error('شناسه نامعتبر.');

        include_once ABSPATH . 'wp-admin/includes/file.php';
        include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

        $url = $this->resolve_download_url($type, $post_id);
        if (!$url) { wp_send_json_error($this->last_error ? $this->last_error : 'لینک دانلود یافت نشد.'); }

        // گزارش شروع
        $this->report_event('install_start', array('type'=>$type,'post_id'=>$post_id,'url'=>$url), 'info');

        $server_referer = rtrim(self::SERVER_BASE, '/').'/';
        $filter_args = function($args, $request_url) use ($server_referer){
            if (!isset($args['headers']) || !is_array($args['headers'])) $args['headers'] = array();
            $args['headers']['User-Agent'] = 'Mozilla/5.0 WP-VPS7Client/' . VPS7_Repository_Client::VERSION;
            $args['headers']['Referer']    = $server_referer;
            $args['headers']['Origin']     = rtrim($server_referer, '/');
            $args['headers']['Accept']     = '*/*';
            $args['redirection']           = max(10, isset($args['redirection']) ? (int)$args['redirection'] : 10);
            $args['timeout']               = max(120, isset($args['timeout']) ? (int)$args['timeout'] : 120);
            $args['reject_unsafe_urls']    = false;
            $args['httpversion']           = '1.1';
            return $args;
        };
        add_filter('http_request_args', $filter_args, 9, 2);

        $curl_tweak = function($handle){
            if (defined('CURL_HTTP_VERSION_1_1')) {
                @curl_setopt($handle, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);
            }
        };
        add_action('http_api_curl', $curl_tweak, 10, 1);

        $http_trace = array();
        $http_debug = function($response, $type, $class, $args, $request_url) use (&$http_trace){
            if ($type === 'response') {
                $code    = wp_remote_retrieve_response_code($response);
                $headers = wp_remote_retrieve_headers($response);
                $loc = '';
                if (is_array($headers)) {
                    $low = array_change_key_case($headers, CASE_LOWER);
                    $loc = $low['location'] ?? '';
                } elseif (is_object($headers) && method_exists($headers,'offsetGet')) {
                    $loc = $headers->offsetGet('location');
                }
                $http_trace[] = array('url'=>$request_url, 'code'=>$code, 'location'=>$loc);
            }
        };
        add_action('http_api_debug', $http_debug, 10, 5);

        $skin = class_exists('WP_Ajax_Upgrader_Skin') ? new WP_Ajax_Upgrader_Skin() : new Automatic_Upgrader_Skin();
        $pkg_opts = function($options){ $options['clear_destination'] = true; $options['abort_if_destination_exists'] = false; return $options; };
        add_filter('upgrader_package_options', $pkg_opts, 10, 1);

        WP_Filesystem();

        $ok=false; $msg='';
        if ($type==='plugin') {
            $up = new Plugin_Upgrader($skin);
            $result = $up->install($url);
            $errors = method_exists($up->skin,'get_errors') ? $up->skin->get_errors() : new WP_Error();
            $ok = $result && (!is_wp_error($result)) && (!is_wp_error($errors) || !$errors->has_errors());
            if (!$ok) { $msg = is_wp_error($result) ? $result->get_error_message() : ( (is_wp_error($errors)&&$errors->has_errors()) ? $errors->get_error_message() : '' ); }
            if (!$ok) {
                $filter_ssl = function($args, $request_url){ $args['sslverify'] = false; return $args; };
                add_filter('http_request_args', $filter_ssl, 11, 2);
                $result = $up->install($url);
                remove_filter('http_request_args', $filter_ssl, 11);
                $errors = method_exists($up->skin,'get_errors') ? $up->skin->get_errors() : new WP_Error();
                $ok = $result && (!is_wp_error($result)) && (!is_wp_error($errors) || !$errors->has_errors());
                $msg = is_wp_error($result) ? $result->get_error_message() : ( (is_wp_error($errors)&&$errors->has_errors()) ? $errors->get_error_message() : $msg );
            }
            if ($ok) wp_clean_plugins_cache(true);
        } else {
            $up = new Theme_Upgrader($skin);
            $result = $up->install($url);
            $errors = method_exists($up->skin,'get_errors') ? $up->skin->get_errors() : new WP_Error();
            $ok = $result && (!is_wp_error($result)) && (!is_wp_error($errors) || !$errors->has_errors());
            if (!$ok) { $msg = is_wp_error($result) ? $result->get_error_message() : ( (is_wp_error($errors)&&$errors->has_errors()) ? $errors->get_error_message() : '' ); }
            if (!$ok) {
                $filter_ssl = function($args, $request_url){ $args['sslverify'] = false; return $args; };
                add_filter('http_request_args', $filter_ssl, 11, 2);
                $result = $up->install($url);
                remove_filter('http_request_args', $filter_ssl, 11);
                $errors = method_exists($up->skin,'get_errors') ? $up->skin->get_errors() : new WP_Error();
                $ok = $result && (!is_wp_error($result)) && (!is_wp_error($errors) || !$errors->has_errors());
                $msg = is_wp_error($result) ? $result->get_error_message() : ( (is_wp_error($errors)&&$errors->has_errors()) ? $errors->get_error_message() : $msg );
            }
            if ($ok) wp_clean_themes_cache();
        }

        remove_filter('http_request_args', $filter_args, 9);
        remove_action('http_api_curl', $curl_tweak, 10);
        remove_action('http_api_debug', $http_debug, 10);
        remove_filter('upgrader_package_options', $pkg_opts, 10);

        if (!$ok) {
            $this->report_event('install_failed', array(
                'type'      => $type,
                'post_id'   => $post_id,
                'url'       => $url,
                'message'   => $msg ?: 'download/install failed',
                'http_trace'=> $http_trace,
            ), 'error');
            wp_send_json_error($msg ? $msg : 'دریافت انجام نشد.');
        }

        $this->report_event('install_ok', array(
            'type'      => $type,
            'post_id'   => $post_id,
            'url'       => $url,
            'http_trace'=> $http_trace,
        ), 'info');

        wp_send_json_success('OK');
    }

    public function ajax_activate_item() {
        check_ajax_referer('vps7_nonce', 'nonce');

        if (!current_user_can('activate_plugins') && !current_user_can('switch_themes')) {
            wp_send_json_error('دسترسی ندارید.');
        }

        $type  = isset($_POST['type'])  ? sanitize_text_field($_POST['type'])  : 'plugin';
        $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : '';
        $slug  = isset($_POST['slug'])  ? sanitize_title($_POST['slug'])       : '';

        if ($type==='plugin') {
            require_once ABSPATH.'wp-admin/includes/plugin.php';
            $file = $this->find_plugin_file_by_slug_or_title($slug, $title);
            if ($file) {
                $r = activate_plugin($file);
                if (is_wp_error($r)) wp_send_json_error($r->get_error_message());
                wp_send_json_success('فعال شد.');
            }
        } else {
            $stylesheet = $this->find_theme_stylesheet_by_slug_or_title($slug, $title);
            if ($stylesheet) { switch_theme($stylesheet); wp_send_json_success('فعال شد.'); }
        }
        wp_send_json_error('مورد یافت نشد.');
    }

    private function find_plugin_file_by_slug_or_title($slug, $title) {
        require_once ABSPATH.'wp-admin/includes/plugin.php';
        $all = get_plugins();
        $cand = array();
        $slug_norm = $slug ? preg_replace('/[\s\-_]+/','', strtolower($slug)) : '';

        foreach ($all as $file=>$data) {
            $folder = explode('/', $file)[0];
            $name   = isset($data['Name']) ? $data['Name'] : '';
            $td     = isset($data['TextDomain']) ? $data['TextDomain'] : '';

            $keys = array(
                sanitize_title($folder),
                sanitize_title($name),
                sanitize_title($td),
                preg_replace('/[\s\-_]+/','', strtolower($folder)),
                preg_replace('/[\s\-_]+/','', strtolower($name)),
                preg_replace('/[\s\-_]+/','', strtolower($td)),
            );

            if ($slug && (sanitize_title($folder) === $slug || preg_replace('/[\s\-_]+/','', strtolower($folder)) === $slug_norm)) {
                return $file;
            }
            $cand[$file] = $keys;
        }

        $title_slug   = $title ? sanitize_title($title) : '';
        $title_norm   = $title ? preg_replace('/[\s\-_]+/','', strtolower($title)) : '';

        foreach ($cand as $file=>$keys) {
            if ($slug && in_array($slug, $keys, true)) return $file;
            if ($slug && $slug_norm && in_array($slug_norm, $keys, true)) return $file;
            if ($title && in_array($title_slug, $keys, true)) return $file;
            if ($title && $title_norm && in_array($title_norm, $keys, true)) return $file;
        }
        return '';
    }

    private function find_theme_stylesheet_by_slug_or_title($slug, $title) {
        $themes = wp_get_themes();
        $slug_norm = $slug ? preg_replace('/[\س\-_]+/','', strtolower($slug)) : '';
        $title_slug = $title ? sanitize_title($title) : '';
        $title_norm = $title ? preg_replace('/[\س\-_]+/','', strtolower($title)) : '';

        foreach ($themes as $stylesheet=>$th) {
            $name = $th->get('Name');
            $keys = array(
                sanitize_title($stylesheet),
                sanitize_title($name),
                preg_replace('/[\س\-_]+/','', strtolower($stylesheet)),
                preg_replace('/[\س\-_]+/','', strtolower($name)),
            );

            if ($slug && (sanitize_title($stylesheet) === $slug || preg_replace('/[\s\-_]+/','', strtolower($stylesheet)) === $slug_norm)) {
                return $stylesheet;
            }
            if ($title && (in_array($title_slug, $keys, true) || in_array($title_norm, $keys, true))) {
                return $stylesheet;
            }
        }
        return '';
    }

    public function ajax_remove_license() {
        check_ajax_referer('vps7_nonce', 'nonce');
        delete_option(self::OPT_LICENSE);
        wp_send_json_success();
    }

    /* ---------------- HTTP helpers ---------------- */
    public function remote_get($url, $args=array()) {
        $url = add_query_arg($args, $url);
        $r = wp_remote_get($url, array('timeout'=>20));
        if (is_wp_error($r)) {
            $this->report_event('remote_get_failed', array(
                'url' => $url,
                'error' => $r->get_error_message(),
            ), 'error');
            return array();
        }
        $b = wp_remote_retrieve_body($r); $d = json_decode($b, true);
        return is_array($d)? $d : array();
    }
    public function remote_post($url, $data=array()) {
        $r = wp_remote_post($url, array('timeout'=>20,'body'=>$data));
        if (is_wp_error($r)) {
            $this->report_event('remote_post_failed', array(
                'url' => $url,
                'error' => $r->get_error_message(),
            ), 'error');
            return array();
        }
        $b = wp_remote_retrieve_body($r); $d = json_decode($b, true);
        return is_array($d)? $d : array();
    }
}

register_activation_hook(__FILE__, function(){ /* no-op */ });
new VPS7_Repository_Client();

/* فایل‌های جانبی اختیاری */
$__je = plugin_dir_path(__FILE__).'je-repository.php';             if ( file_exists($__je) ) require_once $__je;
$__updates = plugin_dir_path(__FILE__).'vps7-client-updates.php';  if ( file_exists($__updates) ) require_once $__updates;
$__repo_admin = plugin_dir_path(__FILE__).'repository-admin.php';  if ( file_exists($__repo_admin) ) require_once $__repo_admin;

/* — خودکار: با 401 یا هدر X-VPS7-Invalidate لایسنس کلاینت پاک شود — */
if (!function_exists('vps7_client_auto_invalidate_license')) {
    function vps7_client_auto_invalidate_license($response, $r, $url){
        if (strpos($url, '/wp-json/vps7/v1/') === false) {
            return $response;
        }
        $code    = (int) wp_remote_retrieve_response_code($response);
        $headers = wp_remote_retrieve_headers($response);
        $body    = wp_remote_retrieve_body($response);

        $invalidate = false;

        if ($code === 401) {
            $invalidate = true;
        }

        $has_header = false;
        if (is_array($headers)) {
            $low = array_change_key_case($headers, CASE_LOWER);
            $has_header = !empty($low['x-vps7-invalidate']);
        } elseif (is_object($headers) && method_exists($headers, 'offsetGet')) {
            $has_header = (bool) $headers->offsetGet('x-vps7-invalidate');
        }
        if ($has_header) $invalidate = true;

        if (!$invalidate && $body) {
            $j = json_decode($body, true);
            if (is_array($j)) {
                if (!empty($j['invalidate'])) $invalidate = true;
                if (!empty($j['code']) && in_array($j['code'], array('license_invalidated','vps7_license_invalidated'), true)) {
                    $invalidate = true;
                }
            }
        }

        if ($invalidate) {
            delete_option('vps7_client_license');
            delete_option('vps7_license_info');
            delete_transient('vps7_license_info');

            add_action('admin_notices', function(){
                echo '<div class="notice notice-error"><p style="font-family:Vazirmatn,sans-serif">لایسنس این سایت توسط سرور آزاد یا نامعتبر شد و از سیستم پاک گردید. لطفاً دوباره لایسنس را ثبت کنید.</p></div>';
            });
        }

        return $response;
    }
    add_filter('http_response', 'vps7_client_auto_invalidate_license', 10, 3);
}

endif;

/* ===========================================================
 * VPS7 Transport Bridge — Global (paste into main plugin file)
 * اثر سراسری روی همهٔ دانلودهای مربوط به /wp-json/vps7/v1/download
 * =========================================================== */
if (!defined('ABSPATH')) exit;
if (!defined('VPS7_CLIENT_TRANSPORT_BRIDGE')) {
    define('VPS7_CLIENT_TRANSPORT_BRIDGE', '1.0.0');

    if (!class_exists('VPS7_Client_Transport_Bridge')) {
        class VPS7_Client_Transport_Bridge {

            public static function boot() {
                add_filter('http_request_args', [__CLASS__, 'http_args'], 11, 2);
                add_filter('upgrader_pre_download', [__CLASS__, 'pre_download'], 10, 3);
                add_filter('http_request_reject_unsafe_urls', [__CLASS__, 'reject_unsafe'], 10, 2);
                add_filter('http_request_host_is_external', [__CLASS__, 'allow_host_when_blocked'], 10, 2);
            }

            protected static function server_base() {
                if (class_exists('VPS7_Repository_Client')) {
                    return rtrim(VPS7_Repository_Client::SERVER_BASE, '/');
                }
                return 'https://vps7.net';
            }
            protected static function server_host() {
                $u = wp_parse_url(self::server_base());
                return !empty($u['host']) ? $u['host'] : 'vps7.net';
            }
            protected static function is_repo_download($url) {
                if (!$url) return false;
                $u = wp_parse_url($url);
                if (empty($u['host']) || empty($u['path'])) return false;
                if (strcasecmp($u['host'], self::server_host()) !== 0) return false;
                return (strpos($u['path'], '/wp-json/vps7/v1/download') !== false);
            }
            public static function http_args($args, $url) {
                if (!self::is_repo_download($url)) return $args;
                if (!isset($args['headers']) || !is_array($args['headers'])) $args['headers'] = array();
                $referer = 'https://' . self::server_host() . '/';
                $args['headers']['User-Agent'] = 'Mozilla/5.0 WP-VPS7Bridge/1.0';
                $args['headers']['Referer']    = $referer;
                $args['headers']['Origin']     = rtrim($referer, '/');
                $args['headers']['Accept']     = '*/*';
                $args['timeout']            = max(180, isset($args['timeout']) ? (int)$args['timeout'] : 0);
                $args['redirection']        = max(10, isset($args['redirection']) ? (int)$args['redirection'] : 0);
                $args['reject_unsafe_urls'] = false;
                $args['httpversion']        = '1.1';
                return $args;
            }
            public static function pre_download($reply, $package, $upgrader) {
                if (!self::is_repo_download($package)) return $reply;
                require_once ABSPATH . 'wp-admin/includes/file.php';
                $tmp = download_url($package, 300);
                if (is_wp_error($tmp)) { return $tmp; }
                return $tmp;
            }
            public static function reject_unsafe($reject, $url) {
                return self::is_repo_download($url) ? false : $reject;
            }
            public static function allow_host_when_blocked($is_external, $host) {
                if (defined('WP_HTTP_BLOCK_EXTERNAL') && WP_HTTP_BLOCK_EXTERNAL) {
                    if (strcasecmp($host, self::server_host()) === 0) return false;
                }
                return $is_external;
            }
        }
    }
    add_action('plugins_loaded', ['VPS7_Client_Transport_Bridge', 'boot'], 1);
}
/* ======================= /VPS7 Transport Bridge ======================= */
